<?php
/**
 * Plugin Name: Tickera - Resend Buyers/Attendees Emails
 * Plugin URI: http://tickera.com/
 * Description: Allows to resend attendees completed order emails for moved events.
 * Author: Tickera.com
 * Author URI: http://tickera.com/
 * Version: 2.2.5
 * Text Domain: tcrbae
 * Domain Path: /languages/
 * Copyright 2015 Tickera (http://tickera.com/)
 */

if ( ! defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

if ( ! class_exists( 'TC_Resend_Buyers_Attendees_Emails' ) ) {

    class TC_Resend_Buyers_Attendees_Emails {

        const VALID_ORDER_STATUSES = [ 'order_paid', 'wc-completed' ];

        var $version = '2.2.5';
        var $title = 'Tickera - Resend Attendees Emails';
        var $name = 'tc-moved-events-resend-emails';
        var $dir_name = 'tc-moved-events-resend-emails';
        var $location = 'plugins';
        var $plugin_dir = '';
        var $plugin_url = '';

        var $page = 1;
        var $per_page = 20;

        function __construct() {
            $this->init_vars();
            require_once $this->plugin_dir . 'includes/general-functions.php';
            add_action( 'activated_plugin', array( $this, 'plugin_dependencies' ), 99 );
            add_filter( 'tc_settings_email_sections', array( $this, 'add_email_section' ), 10, 1 );
            add_filter( 'tc_settings_email_fields', array( $this, 'add_email_section_fields' ), 10, 1 );
            add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
            add_action( 'wp_ajax_tc_resend_attendees_order_emails', array( $this, 'tc_resend_attendees_order_emails' ) );
        }

        /**
         * Load this plugin the very last to ensure all the dependencies loaded first.
         */
        function plugin_dependencies() {

            $plugin_slug = basename( __DIR__ ) . '/' . basename( __FILE__ );
            $active_plugins = get_option( 'active_plugins', [] );
            $index = array_search( $plugin_slug, $active_plugins );

            if ( $index !== false ) {
                unset( $active_plugins[ $index ] );
                $active_plugins[] = $plugin_slug;
                update_option( 'active_plugins', array_values( $active_plugins ) );
            }
        }

        function init_vars() {

            global $tc_plugin_dir, $tc_plugin_url;

            if ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . $this->dir_name . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'subfolder-plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/' . $this->dir_name . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );

            } elseif ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );

            } elseif ( is_multisite() && defined( 'WPMU_PLUGIN_URL' ) && defined( 'WPMU_PLUGIN_DIR' ) && file_exists( WPMU_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'mu-plugins';
                $this->plugin_dir = WPMU_PLUGIN_DIR;
                $this->plugin_url = WPMU_PLUGIN_URL;

            } else {
                wp_die( sprintf( __( 'There was an issue determining where %s is installed. Please reinstall it.', 'tcrbae' ), $this->title ) );
            }

            $tc_plugin_dir = $this->plugin_dir;
            $tc_plugin_url = $this->plugin_url;
        }

        /**
         * Load Assets
         */
        function enqueue_scripts() {
            wp_enqueue_style( $this->name . 'emails-css', plugins_url( 'assets/emails.css', __FILE__ ) );
            wp_enqueue_script( $this->name . 'emails-js', plugins_url( 'assets/emails.js', __FILE__ ) );
            wp_localize_script( $this->name . 'emails-js', 'tc_email_vars', array(
                'ajaxUrl' => apply_filters( 'tc_ajaxurl', admin_url( 'admin-ajax.php', ( is_ssl() ? 'https' : 'http' ) ) ),
                'email_sent' => __( 'Email(s) Sent', 'tcrbae' ),
                'loadingIcon' => $this->plugin_url . '/assets/loading_small.gif',
                'buyers_description' => sprintf( __( 'Body of the e-mail. You can use following placeholders (%s)', 'tcrbae' ), 'EVENT_NAME, EVENT_LOCATION, EVENT_DATE, FIRST_NAME, LAST_NAME, EMAIL, EVENT_TABLE, ORDER_ID, ORDER_DETAILS' ),
                'attendees_description' => sprintf( __( 'Body of the e-mail. You can use following placeholders (%s)', 'tcrbae' ), 'EVENT_NAME, EVENT_LOCATION, EVENT_DATE, FIRST_NAME, LAST_NAME, EMAIL, ORDER_ID, TICKET_TYPE, TICKET_CODE, DOWNLOAD_LINK, ORDER_DETAILS' ),
                'before_send_message' => __( 'The action is irreversible, please make sure you have the correct email content and has been saved.', 'tcrbae' ),
                'report_log_panel_header' => __( 'Email sent to the following addresses:', 'tcrbae' )
            ) );
        }

        /**
         * Additional section under Tickera email settings.
         * Section name: Resend Attendees Order Completed E-Mail
         *
         * @param $sections
         * @return array
         */
        function add_email_section( $sections ) {

            $sections[] = array(
                'name' => 'resend_attendee_order_completed_email',
                'title' => __( 'Resend Buyers/Attendees Order Completed E-Mail' ),
                'description' => 'Resend emails to buyers/attendees. This will notify the buyers/attendees via email with the latest event information. Note: This add-on is not official. Functionality such as sending emails to "All Events" is not yet available.',
            );

            return $sections;
        }

        /**
         * Populate Section Fields.
         * Fields: Event, Order / Ticket Message, Send Email Button
         *
         * @param $default_fields
         * @return array
         */
        function add_email_section_fields( $default_fields ) {

            $resend_attendees_order_emails = array(
                array(
                    'field_name' => 'resend_event_name',
                    'field_title' => __( 'Event', 'tcrbae' ),
                    'field_type' => 'function',
                    'function' => 'tc_get_resend_email_events',
                    'tooltip' => __( 'Select the event to resend attendee email', 'tcrbae' ),
                    'field_description' => '',
                    'table_visibility' => true,
                    'post_field_type' => 'post_meta',
                    'section' => 'resend_attendee_order_completed_email',
                ),
                array(
                    'field_name' => 'resend_event_type',
                    'field_title' => __( 'Send to', 'tcrbae' ),
                    'field_type' => 'function',
                    'function' => 'tc_get_resend_email_types',
                    'tooltip' => __( '', 'tcrbae' ),
                    'field_description' => '',
                    'table_visibility' => true,
                    'post_field_type' => 'post_meta',
                    'section' => 'resend_attendee_order_completed_email',
                ),
                array(
                    'field_name' => 'resend_attendee_order_from_name',
                    'field_title' => __( 'From name', 'tcrbae' ),
                    'field_type' => 'text',
                    'default_value' => tickera_attendee_email_from_name(),
                    'tooltip' => __( '', 'tcrbae' ),
                    'section' => 'resend_attendee_order_completed_email',
                ),
                array(
                    'field_name' => 'resend_attendee_order_from_email',
                    'field_title' => __( 'From email address', 'tcrbae' ),
                    'field_type' => 'text',
                    'default_value' => tickera_attendee_email_from_email(),
                    'tooltip' => __( '', 'tcrbae' ),
                    'section' => 'resend_attendee_order_completed_email',
                ),
                array(
                    'field_name' => 'resend_attendee_order_subject',
                    'field_title' => __( 'Subject', 'tcrbae' ),
                    'field_type' => 'option',
                    'default_value' => __( 'Your Ticket is here!', 'tcrbae' ),
                    'tooltip' => __( 'Subject of the e-mail', 'tcrbae' ),
                    'section' => 'resend_attendee_order_completed_email',
                ),
                array(
                    'field_name' => 'resend_attendee_order_email_content',
                    'field_title' => __( 'Order / Ticket Message', 'tcrbae' ),
                    'field_type' => 'function',
                    'function' => 'tickera_get_attendee_order_message',
                    'default_value' => 'Hello, <br /><br />You can download ticket for EVENT_NAME here DOWNLOAD_LINK',
                    'field_description' => sprintf( __( 'Body of the e-mail. You can use following placeholders (%s)', 'tcrbae' ), 'EVENT_NAME, EVENT_LOCATION, EVENT_DATE, FIRST_NAME, LAST_NAME, EMAIL, EVENT_TABLE, ORDER_ID, ORDER_DETAILS' ),
                    'section' => 'resend_attendee_order_completed_email',
                ),
            );

            $default_fields = array_merge( $default_fields, $resend_attendees_order_emails );

            return $default_fields;
        }

        /**
         * Collect orders and attendees data
         */
        function tc_resend_attendees_order_emails() {

            if ( isset( $_POST[ 'event_id' ] ) ) {

                $sent_emails = [];

                $event_id = (int) $_POST[ 'event_id' ];
                $send_to = sanitize_text_field( $_POST[ 'send_to' ] );

                if ( $event_id ) {

                    add_filter( 'wp_mail_content_type', function( $content_type ) {
                        return 'text/html';
                    } );

                    global $wpdb;
                    $query = $wpdb->prepare( "SELECT p.ID, p.post_parent as order_id FROM {$wpdb->posts} p, {$wpdb->postmeta} pm WHERE p.ID = pm.post_id AND pm.meta_key = 'event_id' AND pm.meta_value = %d", $event_id );
                    $ticket_instances = $wpdb->get_results( $query );

                    // Prepare Order Ids
                    $order_ids = [];
                    foreach ( $ticket_instances as $ticket_instance ) {
                        $order_id = $ticket_instance->order_id;
                        if ( in_array( get_post_status( $order_id ), self::VALID_ORDER_STATUSES ) &&  'trash' != get_post_status( $ticket_instance->ID ) ) {
                            $order_ids[ $order_id ][] = $ticket_instance->ID;
                        }
                    }

                    foreach ( $order_ids as $order_id => $ticket_instances ) {

                        switch ( $send_to ) {

                            case 'buyers':
                                $sent = self::send_to_buyer( $order_id, $ticket_instances );
                                if ( $sent ) $sent_emails[] = $sent;
                                break;

                            case 'attendees':
                                $sent = self::send_to_attendee( $ticket_instances );
                                if ( $sent ) $sent_emails = array_merge( $sent_emails, $sent );
                                break;
                        }
                    }
                }

                wp_send_json( [ 'count' => count( $sent_emails ), 'sent_emails' => $sent_emails ] );
            }
        }

        /**
         * Resend event details directly to the buyers email.
         *
         * @param $order_id
         * @param $ticket_instances
         * @return bool
         */
        function send_to_buyer( $order_id, $ticket_instances ) {

            $event_id = (int) $_POST[ 'event_id' ];

            if ( ! $event_id ) {
                return false;
            }

            $event_arr = []; $ticket_data = [];
            $event = new Tickera\TC_Event( $event_id );

            // Prepare Event Data
            foreach ( $ticket_instances as $ticket_instance_id ) {

                $ticket_type_id = get_post_meta( $ticket_instance_id, 'ticket_type_id', true );

                $event_arr = [
                    'title'             => $event->details->post_title,
                    'event_location'    => get_post_meta( $event_id, 'event_location', true ),
                    'event_date'        => $event->get_event_date()
                ];

                $ticket_data[] = [
                    'first_name'    => get_post_meta( $ticket_instance_id, 'first_name', true ),
                    'last_name'     => get_post_meta( $ticket_instance_id, 'last_name', true ),
                    'email'         => get_post_meta( $ticket_instance_id, 'owner_email', true ),
                    'ticket_type'   => get_the_title( $ticket_type_id ),
                    'ticket_code'   => get_post_meta( $ticket_instance_id, 'ticket_code', true ),
                    'download_link' => tickera_get_ticket_download_link( '', '', $ticket_instance_id, true )
                ];

                $event_arr[ 'ticket_data' ] = $ticket_data;
            }

            if ( self::is_bridge( $order_id ) ) {

                // Bridge for Woocommerce
                $order = wc_get_order( $order_id );
                $order_title = $order_id;
                $buyer_first_name = $order->get_billing_first_name();
                $buyer_last_name = $order->get_billing_last_name();
                $buyer_email = $order->get_billing_email();

                $order_date = $order->get_date_created();
                $order_status = $order->get_status();

            } else {

                // Tickera Standalone
                $buyer_data = get_post_meta( $order_id, 'tc_cart_info', true );
                $buyer_data = $buyer_data['buyer_data'] ? $buyer_data['buyer_data'] : [];
                $buyer_first_name = $buyer_data['first_name_post_meta'] ? $buyer_data['first_name_post_meta'] : '';
                $buyer_last_name = $buyer_data['last_name_post_meta'] ? $buyer_data['last_name_post_meta'] : '';
                $buyer_email = $buyer_data['email_post_meta'] ? $buyer_data['email_post_meta'] : '';
                $order_title = get_the_title( $order_id );

                $order = new \Tickera\TC_Order( $order_id );
                $order_date = $order->details->tc_order_date;
                $order_status = $order->details->post_status;
            }

            $order_details = tickera_get_order_details_email( $order_id, $order_date, true, $order_status );
            $placeholders = array( 'EVENT_NAME', 'EVENT_LOCATION', 'EVENT_DATE', 'FIRST_NAME','LAST_NAME', 'EMAIL', 'EVENT_TABLE', 'ORDER_ID', 'ORDER_DETAILS' );
            $placeholder_values = array(
                $event_arr[ 'title' ],
                $event_arr[ 'event_location' ],
                $event_arr[ 'event_date' ],
                $buyer_first_name,
                $buyer_last_name,
                $buyer_email,
                tc_render_event_table( $event_arr ),
                $order_title,
                $order_details
            );

            $email_settings = get_option( 'tickera_email_setting', false );
            $content = $email_settings[ 'resend_attendee_order_email_content' ];
            $content = $content ? $content : 'Hello, <br /><br />You can download the tickets below <br/> EVENT_TABLE';
            $content = str_replace( $placeholders, $placeholder_values, $content );
            return self::send_mail( $buyer_email, $content );
        }

        /**
         * Resend event details directly to the attendees email.
         *
         * @param $ticket_instances
         * @return bool
         */
        function send_to_attendee( $ticket_instances ) {

            $sent_emails = [];
            $event_id = (int) $_POST[ 'event_id' ];

            if ( ! $event_id ) {
                return false;
            }

            $event = new Tickera\TC_Event( $event_id );

            foreach ( $ticket_instances as $ticket_instance_id ) {

                $order_id = wp_get_post_parent_id( $ticket_instance_id );

                $order = new \Tickera\TC_Order( $order_id );
                $order_title = get_the_title( $order_id );
                $order_date = $order->details->tc_order_date;
                $order_status = $order->details->post_status;

                if ( self::is_bridge( $order_id ) ) {
                    $order_title = $order_id;
                    $order = wc_get_order( $order_id );
                    $order_date = $order->get_date_created();
                    $order_status = $order->get_status();
                }

                $event_title = $event->details->post_title;
                $event_location = get_post_meta( $event_id, 'event_location', true );
                $event_date = $event->get_event_date();

                $first_name = get_post_meta( $ticket_instance_id, 'first_name', true );
                $last_name = get_post_meta( $ticket_instance_id, 'last_name', true );
                $email = get_post_meta( $ticket_instance_id, 'owner_email', true );

                $ticket_type_id = get_post_meta( $ticket_instance_id, 'ticket_type_id', true );
                $ticket_type = get_the_title( $ticket_type_id );
                $ticket_code = get_post_meta( $ticket_instance_id, 'ticket_code', true );
                $download_link = tickera_get_ticket_download_link( '', '', $ticket_instance_id, true );
                $order_details = tickera_get_order_details_email( $order_id, $order_date, true, $order_status );

                $placeholders = array( 'EVENT_NAME', 'EVENT_LOCATION', 'EVENT_DATE', 'FIRST_NAME','LAST_NAME', 'EMAIL', 'ORDER_ID', 'TICKET_TYPE', 'TICKET_CODE','DOWNLOAD_LINK', 'ORDER_DETAILS' );
                $placeholder_values = array(
                    $event_title,
                    $event_location,
                    $event_date,
                    $first_name,
                    $last_name,
                    $email,
                    $order_title,
                    $ticket_type,
                    $ticket_code,
                    $download_link,
                    $order_details
                );

                $email_settings = get_option( 'tickera_email_setting', false );
                $content = $email_settings[ 'resend_attendee_order_email_content' ];
                $content = $content ? $content : 'Hello, <br /><br />You can get your ticket here <br/> DOWNLOAD_LINK';
                $content = str_replace( $placeholders, $placeholder_values, $content );
                $sent = self::send_mail( $email, $content );
                if ( $sent ) $sent_emails[] = $sent;
            }

            return $sent_emails;
        }

        /**
         * Send Email
         *
         * @param $email
         * @param $content
         * @return bool
         */
        function send_mail( $email, $content ) {

            if ( ! $email ) {
                return false;
            }

            $tc_email_settings = get_option( 'tickera_email_setting', false );
            $tc_email_send_type = isset( $tc_email_settings[ 'email_send_type' ] ) ? $tc_email_settings[ 'email_send_type' ] : 'wp_mail';
            $subject = isset( $tc_email_settings[ 'resend_attendee_order_subject' ] ) ? stripslashes( $tc_email_settings[ 'resend_attendee_order_subject' ] ) : __( 'Your Ticket is here!', 'tcrbae' );
            $from_name = isset( $tc_email_settings[ 'resend_attendee_order_from_name' ] ) ? $tc_email_settings[ 'resend_attendee_order_from_name' ] : tickera_attendee_email_from_name();
            $from_email = isset( $tc_email_settings[ 'resend_attendee_order_from_email' ] ) ? $tc_email_settings[ 'resend_attendee_order_from_email' ] : tickera_attendee_email_from_email();;

            if ( 'wp_mail' == $tc_email_send_type ) {

                $headers = array(
                    'From: ' . $from_name . ' <' . $from_email . '>',
                    'Content-Type: text/html; charset=UTF-8'
                );

                wp_mail( $email, $subject, html_entity_decode( stripcslashes( wpautop( $content ) ) ), $headers );

            } else {

                $headers = 'MIME-Version: 1.0' . "\r\n";
                $headers .= 'Content-type: text/html; charset=UTF-8' . "\r\n";
                $headers .= 'From: ' . $from_email . "\r\n" .
                    'Reply-To: ' . $from_email . "\r\n" .
                    'X-Mailer: PHP/' . phpversion();

                mail( $email, $subject, stripcslashes( wpautop( $content ) ), $headers );
            }

            return $email;
        }

        /**
         * Check if Bridge for Woocommerce is active or not
         *
         * @param $order_id
         * @return bool
         */
        function is_bridge( $order_id ) {
            return ( 'shop_order' == get_post_type( $order_id ) ) ? true : false;
        }
    }

    $TC_Resend_Buyers_Attendees_Emails = new TC_Resend_Buyers_Attendees_Emails();
}
