<?php
/**
 * Plugin Name: Tickera - Create order and tickets from imported CSV
 * Plugin URI: https://tickera.com/
 * Description: Create order and tickets from imported CSV ( Custom add-on for Tickera )
 * Version: 2.7
 * Author: Tickera
 * Author URI: https://tickera.com/
 * Developer: Tickera
 * Developer URI: https://tickera.com/
 */
if ( ! defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

if ( ! class_exists( 'TC_Import_Order_Tickets_From_CSV' ) ) {

    class TC_Import_Order_Tickets_From_CSV {

		var $version = '2.7';
		var $title = 'Tickera - Create order and tickets from imported CSV';
		var $name = 'tc-custom-create-orders-and-tickets-from-csv';
		var $dir_name = 'tc-custom-create-orders-and-tickets-from-csv';
		var $location = 'plugins';
		var $plugin_dir = '';
		var $plugin_url = '';

        function __construct() {
			self::init_vars();
            self::maybe_create_uploads_dir();
            add_action( 'activated_plugin', array( $this, 'plugin_dependencies' ), 99 );
            add_filter( 'tc_settings_new_menus', array( $this, 'tc_settings_new_menus_custom_csv_import' ) );
            add_action( 'tc_settings_menu_tickera_custom_import_csv', array( $this, 'tc_settings_menu_tickera_custom_import_csv_show_page' ) );
            add_action( 'admin_init', array( $this, 'tc_custom_import_data' ), 0 );
        }

		function init_vars() {

			if ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . $this->dir_name . '/' . basename( __FILE__ ) ) ) {
				$this->location = 'subfolder-plugins';
				$this->plugin_dir = WP_PLUGIN_DIR . '/' . $this->dir_name . '/';
				$this->plugin_url = plugins_url( '/', __FILE__ );

			} elseif ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
				$this->location = 'plugins';
				$this->plugin_dir = WP_PLUGIN_DIR . '/';
				$this->plugin_url = plugins_url( '/', __FILE__ );

			} elseif ( is_multisite() && defined( 'WPMU_PLUGIN_URL' ) && defined( 'WPMU_PLUGIN_DIR' ) && file_exists( WPMU_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
				$this->location = 'mu-plugins';
				$this->plugin_dir = WPMU_PLUGIN_DIR;
				$this->plugin_url = WPMU_PLUGIN_URL;
			} else {
				wp_die( sprintf( __( 'There was an issue determining where %s is installed. Please reinstall it.', 'tc' ), $this->title ) );
			}
		}

        /**
         * Load this plugin the very last to ensure all the dependencies loaded first.
         */
        function plugin_dependencies() {

            $plugin_slug = basename( __DIR__ ) . '/' . basename( __FILE__ );
            $active_plugins = get_option( 'active_plugins', [] );
            $index = array_search( $plugin_slug, $active_plugins );

            if ( $index !== false ) {
                unset( $active_plugins[ $index ] );
                $active_plugins[] = $plugin_slug;
                update_option( 'active_plugins', array_values( $active_plugins ) );
            }
        }

		function tc_settings_new_menus_custom_csv_import( $settings_tabs ) {
			$settings_tabs[ 'tickera_custom_import_csv' ] = __( 'Import CSV', 'tc' );
			return $settings_tabs;
		}

		function tc_settings_menu_tickera_custom_import_csv_show_page() {
			require_once $this->plugin_dir . 'includes/admin-pages/settings-tickera_custom_import_csv.php';
		}

        function tc_custom_import_data() {

            if ( isset( $_POST[ 'tc_custom_import_csv' ] ) ) {

                @error_reporting( E_ERROR );
                @set_time_limit( 0 );
                @ini_set( 'max_input_time', 3600 * 3 );
                @ini_set( 'max_execution_time', 3600 * 3 );

                if ( $_FILES[ 'tc_csv_import_file' ][ 'size' ] > 0 && 'text/csv' == stripslashes( $_FILES[ 'tc_csv_import_file' ][ 'type' ] ) ) {

                    // Get the csv file
                    $file = $_FILES[ 'tc_csv_import_file' ][ 'tmp_name' ];
                    $handle = fopen( $file, "r" );

                    // Generate report csv file
                    $upload = wp_upload_dir();
                    $upload_dir = $upload['basedir'];
                    $upload_dir = $upload_dir . '/tc-orders-from-csv/';

                    $report_file = $upload_dir . 'orders.csv';
                    @chmod( $report_file, 0644 );
                    $report_csv = @fopen( $report_file, 'w' );
                    fputcsv( $report_csv, [ 'buyer_first_name', 'buyer_last_name', 'buyer_email', 'order_id', 'ticket_codes', 'status', 'remarks' ] );

                    $row = 0;

                    while ( ! feof( $handle ) ) {

                        $data = fgetcsv( $handle, 100000, ",", "'" );

                        if ( $row && $data ) {

                            $fields = apply_filters( 'tc_import_order_ticket_fields', [
                                'buyer_first_name' => $data[0],
                                'buyer_last_name' => $data[1],
                                'buyer_email' => $data[2],
                                'owner_first_name' => $data[3],
                                'owner_last_name' => $data[4],
                                'owner_email' => $data[5],
                                'ticket_code' => $data[6],
                                'order_id' => $data[7],
                            ] );

                            $result = self::tc_custom_create_order_from_csv( (int) $_POST[ 'tc_ticket_type_id' ], $fields );

                            // Prepare report data
                            $report_data = [
                                $fields[ 'buyer_first_name' ],
                                $fields[ 'buyer_last_name' ],
                                $fields[ 'buyer_email' ],
                                $fields[ 'order_id' ],
                                $fields[ 'ticket_code' ],
                                $result[ 'created' ] ? 'created' : 'failed',
                                $result[ 'error' ] ? $result[ 'error' ] : ''
                            ];

                            fputcsv( $report_csv, $report_data );
                        }

                        $row++;
                    }

                    fclose( $handle );
                    fclose( $report_csv );
                }
            }
        }

        function tc_custom_create_order_from_csv( $tc_ticket_type_id, $fields ) {

            global $tc;

            $order_id = isset( $fields['order_id'] ) ? $fields['order_id'] : '';

            // Order ID
            if ( ! $order_id ) {
                $order_id = $tc->generate_order_id();
            }

            require_once( $tc->plugin_dir . 'includes/classes/class.payment_gateways.php' );
            require_once( $tc->plugin_dir . 'includes/gateways/free-orders.php' );

            $gateway_class = 'TC_Gateway_Free_Orders';
            $free_orders_gateway = new Tickera\Gateway\TC_Gateway_Free_Orders();

            // Cart Contents
            $cart = [];
            $cart[ $tc_ticket_type_id ] = 1;
            $cart_contents = $cart;

            // Buyer Data
            $buyer_data = [];
            $buyer_data[ 'first_name_post_meta' ] = $fields['buyer_first_name'];
            $buyer_data[ 'last_name_post_meta' ] = $fields['buyer_last_name'];
            $buyer_data[ 'email_post_meta' ] = $fields['buyer_email'];

            // Owner Data
            $owner_data = [];
            $owner_data[ 'first_name_post_meta' ] = explode( '|', $fields['owner_first_name'] );
            $owner_data[ 'last_name_post_meta' ] = explode( '|', $fields['owner_last_name'] );
            $owner_data[ 'email_post_meta' ] = explode( '|', $fields['owner_email'] );
            $owner_data[ 'ticket_type_id_post_meta' ] = $tc_ticket_type_id;
            $owner_data[ 'ticket_code_post_meta' ] = explode( '|', $fields['ticket_code'] );

            // Cart Info
            $cart_info = [];
            $cart_info[ 'coupon_code' ] = '';
            $cart_info[ 'total' ] = 0;
            $cart_info[ 'currency' ] = $tc->get_cart_currency();
            $cart_info[ 'buyer_data' ] = $buyer_data;
            $cart_info[ 'owner_data' ] = $owner_data;
            $cart_info[ 'gateway_class' ] = $gateway_class;
            $cart_info[ 'gateway' ] = $free_orders_gateway->plugin_name;
            $cart_info[ 'gateway_admin_name' ] = $free_orders_gateway->admin_name;

            // Payment Info
            $payment_info = [];
            $payment_info[ 'gateway_public_name' ] = $free_orders_gateway->public_name;
            $payment_info[ 'gateway_private_name' ] = $free_orders_gateway->admin_name;
            $payment_info[ 'method' ] = $free_orders_gateway->admin_name;
            $payment_info[ 'total' ] = 0;
            $payment_info[ 'subtotal' ] = 0;
            $payment_info[ 'discount' ] = 0;
            $payment_info[ 'fees_total' ] = 0;
            $payment_info[ 'tax_total' ] = 0;
            $payment_info[ 'currency' ] = $tc->get_cart_currency();

            return self::tc_custom_csv_create_order_internal( $order_id, $cart_contents, $cart_info, $payment_info, true );
        }

        function tc_custom_csv_create_order_internal( $order_id, $cart_contents, $cart_info, $payment_info, $paid ) {

            global $tc;

            if ( empty( $order_id ) ) {
                $order_id = $tc->generate_order_id();

            } elseif ( self::get_order( $order_id ) ) {
                // Don't continue if the order exists
                return [ 'error' => 'OrderID already exists', 'created' => false ];
            }

            $remarks = '';
            $owner_data = $cart_info[ 'owner_data' ];

            // Validate ticket codes
            $ticket_codes_exists = self::validate_codes( $owner_data[ 'ticket_code_post_meta' ] );
            if ( $ticket_codes_exists ) {
                return [ 'error' => $ticket_codes_exists, 'created' => false ];
            }

            $cart_total = $cart_info[ 'total' ];
            $fraud = $tc->check_for_total_paid_fraud( $payment_info[ 'total' ], $cart_total );
            $user_id = get_current_user_id();

            // Insert post type
            $status = ( $paid ? ( $fraud ? 'order_fraud' : 'order_paid' ) : 'order_received' );

            $order = array();
            $order[ 'post_title' ] = $order_id;
            $order[ 'post_name' ] = $order_id;
            $order[ 'post_content' ] = serialize( $cart_contents );
            $order[ 'post_status' ] = $status;
            $order[ 'post_type' ] = 'tc_orders';

            if ( $user_id != 0 ) {
                $order[ 'post_author' ] = $user_id;
            }

            $post_id = wp_insert_post( $order );

            // Cart Contents
            add_post_meta( $post_id, 'tc_cart_contents', $cart_contents );

            // Cart Info
            add_post_meta( $post_id, 'tc_cart_info', $cart_info ); //save row data - buyer and ticket owners data, gateway, total, currency, coupon code, etc.

            // Payment Info
            add_post_meta( $post_id, 'tc_payment_info', $payment_info ); //transaction_id, total, currency, method

            // Order Date & Time
            add_post_meta( $post_id, 'tc_order_date', time() );

            // Order Paid Time
            add_post_meta( $post_id, 'tc_paid_date', ( $paid ) ? time() : '' ); //empty means not yet paid

            // Event(s) - could be more events at once since customer may have tickets from more than one event in the cart
            add_post_meta( $post_id, 'tc_parent_event', $tc->get_cart_events( $cart_contents ) );
            add_post_meta( $post_id, 'tc_event_creators', $tc->get_events_creators( $cart_contents ) );

            // Discount Code
            add_post_meta( $post_id, 'tc_paid_date', ( $paid ) ? time() : '' );

            // Save Ticket Owner(s) data
            $owner_count = max( [ count( $owner_data[ 'first_name_post_meta' ] ), count( $owner_data[ 'last_name_post_meta' ] ), count( $owner_data[ 'email_post_meta' ] ), count( $owner_data[ 'ticket_code_post_meta' ] ) ] );

            $missing_ticket_codes = [];
            for ( $i = 0; $i < $owner_count; $i++ ) {

                $metas[ 'first_name' ] = ( isset( $owner_data[ 'first_name_post_meta' ][ $i ] ) ) ? $owner_data[ 'first_name_post_meta' ][ $i ] : '';
                $metas[ 'last_name' ] = ( isset( $owner_data[ 'last_name_post_meta' ][ $i ] ) ) ? $owner_data[ 'last_name_post_meta' ][ $i ] : '';
                $metas[ 'owner_email' ] = ( isset( $owner_data[ 'email_post_meta' ][ $i ] ) ) ? $owner_data[ 'email_post_meta' ][ $i ] : '';
                $metas[ 'ticket_type_id' ] = $owner_data[ 'ticket_type_id_post_meta' ];

                $ticket_type = new Tickera\TC_Ticket( $metas[ 'ticket_type_id' ] );
                $event_id = $ticket_type->get_ticket_event();
                $metas[ 'event_id' ] = $event_id;

                $arg = array(
                    'post_author' => isset( $user_id ) ? $user_id : '',
                    'post_parent' => $post_id,
                    'post_excerpt' => ( isset( $excerpt ) ? $excerpt : '' ),
                    'post_content' => ( isset( $content ) ? $content : '' ),
                    'post_status' => 'publish',
                    'post_title' => ( isset( $title ) ? $title : '' ),
                    'post_type' => 'tc_tickets_instances',
                );

                $owner_record_id = @wp_insert_post( $arg, true );

                if ( isset( $owner_data[ 'ticket_code_post_meta' ] ) ) {

                    if ( isset( $owner_data[ 'ticket_code_post_meta' ][ $i ] ) && $owner_data[ 'ticket_code_post_meta' ][ $i ] ) {
                        $metas[ 'ticket_code' ] = $owner_data[ 'ticket_code_post_meta' ][ $i ];

                    } else {

                        if ( apply_filters( 'tc_use_only_digit_order_number', false ) == true ) {
                            $metas[ 'ticket_code' ] = apply_filters( 'tc_ticket_code', $order_id . '' . ( $i + 1 ), $owner_data[ 'ticket_type_id_post_meta' ] );

                        } else {
                            $metas[ 'ticket_code' ] = apply_filters( 'tc_ticket_code', $order_id . '-' . ( $i + 1 ), $owner_data[ 'ticket_type_id_post_meta' ] );
                        }

                        $missing_ticket_codes[ $i ] = $metas[ 'ticket_code' ];
                    }

                } else {

                    if ( apply_filters( 'tc_use_only_digit_order_number', false ) == true ) {
                        $metas[ 'ticket_code' ] = apply_filters( 'tc_ticket_code', $order_id . '' . ( $i + 1 ), $owner_data[ 'ticket_type_id_post_meta' ] );

                    } else {
                        $metas[ 'ticket_code' ] = apply_filters( 'tc_ticket_code', $order_id . '-' . ( $i + 1 ), $owner_data[ 'ticket_type_id_post_meta' ] );
                    }
                }

                foreach ( $metas as $meta_name => $mata_value ) {
                    update_post_meta( $owner_record_id, $meta_name, $mata_value );
                }
            }

            // Collection of remarks
            $remarks .= ( $missing_ticket_codes ) ? 'TicketCode(s) [' . implode( '],[', array_keys( $missing_ticket_codes ) ) . '] missing. Replaced with default code(s) [' . implode( '],[', $missing_ticket_codes ) . '].' : '';

            // Send order status email to the customer
            do_action( 'tc_order_created', $order_id, $status, $cart_contents, $cart_info, $payment_info );
            $tc->session->drop( 'tc_order' );
            return [ 'error' => ( $post_id ? $remarks : 'OrderID already exists' ), 'created' => ( $post_id ? true : false ) ];
        }

        function get_order( $order_id ) {

            $id = ( is_int( $order_id ) ) ? $order_id : tickera_get_order_id_by_name( $order_id );

            if ( empty( $id ) )
                return false;

            $order = get_post( $id->ID );

            if ( ! $order )
                return false;

            $meta = get_post_custom( $id->ID );

            foreach ( $meta as $key => $val )
                $order->{$key} = maybe_unserialize( $meta[ $key ][ 0 ] );

            return $order;
        }

        function maybe_create_uploads_dir() {

            try {
                $upload = wp_upload_dir();
                $upload_dir = $upload['basedir'];
                $upload_dir = $upload_dir . '/tc-orders-from-csv';

                if (!is_dir($upload_dir)) {
                    @mkdir($upload_dir, 0755);
                }

            } catch (Exception $e) {
                // We can't make a directory
            }
        }

        function validate_codes( $codes ) {

            if ( ! is_array( $codes ) ) {
                return;
            }

            $tickets = get_posts( [
                'post_type' => 'tc_tickets_instances',
                'meta_key' => 'ticket_code',
                'meta_value' => $codes,
                'fields' => 'ids',
                'posts_per_page' => -1
            ]);

            $exists = [];
            foreach ( $tickets as $ticket_id ) {
                $exists[] = get_post_meta( $ticket_id, 'ticket_code', true );
            }

            return $exists ? 'Ticket code(s) ' . implode( ', ', $exists ) . ' already exists.' : false;
        }
    }

    $TC_Import_Order_Tickets_From_CSV = new TC_Import_Order_Tickets_From_CSV();
}
